/**
 * @name string-match-left-right
 * @fileoverview Match substrings on the left or right of a given index, ignoring whitespace
 * @version 7.1.0
 * @author Roy Revelt, Codsen Ltd
 * @license MIT
 * {@link https://codsen.com/os/string-match-left-right/}
 */

(function (global, factory) {
typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
typeof define === 'function' && define.amd ? define(['exports'], factory) :
(global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.stringMatchLeftRight = {}));
}(this, (function (exports) { 'use strict';

/**
 * @name arrayiffy-if-string
 * @fileoverview Put non-empty strings into arrays, turn empty-ones into empty arrays. Bypass everything else.
 * @version 3.14.0
 * @author Roy Revelt, Codsen Ltd
 * @license MIT
 * {@link https://codsen.com/os/arrayiffy-if-string/}
 */

function arrayiffy(something) {
  if (typeof something === "string") {
    if (something.length) {
      return [something];
    }
    return [];
  }
  return something;
}

/* eslint no-plusplus:0 */
function isObj(something) {
    return (something && typeof something === "object" && !Array.isArray(something));
}
function isStr(something) {
    return typeof something === "string";
}
const defaults = {
    cb: undefined,
    i: false,
    trimBeforeMatching: false,
    trimCharsBeforeMatching: [],
    maxMismatches: 0,
    firstMustMatch: false,
    lastMustMatch: false,
    hungry: false,
};
const defaultGetNextIdx = (index) => index + 1;
// eslint-disable-next-line consistent-return
function march(str, position, whatToMatchVal, originalOpts, special = false, getNextIdx = defaultGetNextIdx) {
    const whatToMatchValVal = typeof whatToMatchVal === "function" ? whatToMatchVal() : whatToMatchVal;
    // early ending case if matching EOL being at 0-th index:
    if (+position < 0 && special && whatToMatchValVal === "EOL") {
        return whatToMatchValVal;
    }
    const opts = { ...defaults, ...originalOpts };
    if (position >= str.length && !special) {
        return false;
    }
    // The "charsToCheckCount" varies, it decreases with skipped characters,
    // as long as "maxMismatches" allows. It's not the count of how many
    // characters de-facto have been matched from the source.
    let charsToCheckCount = special ? 1 : whatToMatchVal.length;
    // this is the counter of real characters matched. It is not reduced
    // from the holes in matched. For example, if source is "abc" and
    // maxMismatches=1 and we have "ac", result of the match will be true,
    // the following var will be equal to 2, meaning we matched two
    // characters:
    let charsMatchedTotal = 0;
    // used to catch frontal false positives, where too-eager matching
    // depletes the mismatches allowance before precisely matching the exact
    // string that follows, yielding too early false-positive start
    let patienceReducedBeforeFirstMatch = false;
    let lastWasMismatched = false; // value is "false" or index of where it was activated
    // if no character was ever matched, even through if opts.maxMismatches
    // would otherwise allow to skip characters, this will act as a last
    // insurance - at least one character must have been matched to yield a
    // positive result!
    let atLeastSomethingWasMatched = false;
    let patience = opts.maxMismatches;
    let i = position;
    // internal-use flag, not the same as "atLeastSomethingWasMatched":
    let somethingFound = false;
    // these two drive opts.firstMustMatch and opts.lastMustMatch:
    let firstCharacterMatched = false;
    let lastCharacterMatched = false;
    // bail early if there's whitespace in front, imagine:
    // abc important}
    //   ^
    //  start, match ["!important"], matchRightIncl()
    //
    // in case above, "c" consumed 1 patience, let's say 1 is left,
    // we stumble upon "i" where "!" is missing. "c" is false start.
    function whitespaceInFrontOfFirstChar() {
        return (
        // it's a first letter match
        charsMatchedTotal === 1 &&
            // and character in front exists
            // str[i - 1] &&
            // and it's whitespace
            // !str[i - 1].trim() &&
            // some patience has been consumed already
            patience < opts.maxMismatches - 1);
    }
    while (str[i]) {
        const nextIdx = getNextIdx(i);
        if (opts.trimBeforeMatching && str[i].trim() === "") {
            if (!str[nextIdx] && special && whatToMatchVal === "EOL") {
                return true;
            }
            i = getNextIdx(i);
            continue;
        }
        if ((opts &&
            !opts.i &&
            opts.trimCharsBeforeMatching &&
            opts.trimCharsBeforeMatching.includes(str[i])) ||
            (opts &&
                opts.i &&
                opts.trimCharsBeforeMatching &&
                opts.trimCharsBeforeMatching
                    .map((val) => val.toLowerCase())
                    .includes(str[i].toLowerCase()))) {
            if (special && whatToMatchVal === "EOL" && !str[nextIdx]) {
                // return true because we reached the zero'th index, exactly what we're looking for
                return true;
            }
            i = getNextIdx(i);
            continue;
        }
        const charToCompareAgainst = nextIdx > i
            ? whatToMatchVal[whatToMatchVal.length - charsToCheckCount]
            : whatToMatchVal[charsToCheckCount - 1];
        // let's match
        if ((!opts.i && str[i] === charToCompareAgainst) ||
            (opts.i && str[i].toLowerCase() === charToCompareAgainst.toLowerCase())) {
            if (!somethingFound) {
                somethingFound = true;
            }
            if (!atLeastSomethingWasMatched) {
                atLeastSomethingWasMatched = true;
            }
            // if this was the first character from the "to-match" list, flip the flag
            if (charsToCheckCount === whatToMatchVal.length) {
                firstCharacterMatched = true;
                // now, if the first character was matched and yet, patience was
                // reduced already, this means there's a false beginning in front
                if (patience !== opts.maxMismatches) {
                    return false;
                }
            }
            else if (charsToCheckCount === 1) {
                lastCharacterMatched = true;
            }
            charsToCheckCount -= 1;
            charsMatchedTotal++;
            // bail early if there's whitespace in front, imagine:
            // abc important}
            //   ^
            //  start, match ["!important"], matchRightIncl()
            //
            // in case above, "c" consumed 1 patience, let's say 1 is left,
            // we stumble upon "i" where "!" is missing. "c" is false start.
            if (whitespaceInFrontOfFirstChar()) {
                return false;
            }
            if (!charsToCheckCount) {
                return (
                // either it was not a perfect match
                charsMatchedTotal !== whatToMatchVal.length ||
                    // or it was, and in that case, no patience was reduced
                    // (if a perfect match was found, yet some "patience" was reduced,
                    // that means we have false positive characters)
                    patience === opts.maxMismatches ||
                    // mind you, it can be a case of rogue characters in-between
                    // the what was matched, imagine:
                    // source: "abxcd", matching ["bc"], maxMismatches=1
                    // in above case, charsMatchedTotal === 2 and whatToMatchVal ("bc") === 2
                    // - we want to exclude cases of frontal false positives, like:
                    // source: "xy abc", match "abc", maxMismatches=2, start at 0
                    //          ^
                    //       match form here to the right
                    !patienceReducedBeforeFirstMatch
                    ? i
                    : false);
            }
        }
        else {
            if (!patienceReducedBeforeFirstMatch && !charsMatchedTotal) {
                patienceReducedBeforeFirstMatch = true;
            }
            if (opts.maxMismatches && patience && i) {
                patience -= 1;
                // the bigger the maxMismatches, the further away we must check for
                // alternative matches
                for (let y = 0; y <= patience; y++) {
                    // maybe str[i] will match against next charToCompareAgainst?
                    const nextCharToCompareAgainst = nextIdx > i
                        ? whatToMatchVal[whatToMatchVal.length - charsToCheckCount + 1 + y]
                        : whatToMatchVal[charsToCheckCount - 2 - y];
                    const nextCharInSource = str[getNextIdx(i)];
                    if (nextCharToCompareAgainst &&
                        ((!opts.i && str[i] === nextCharToCompareAgainst) ||
                            (opts.i &&
                                str[i].toLowerCase() ===
                                    nextCharToCompareAgainst.toLowerCase())) &&
                        // ensure we're not skipping the first enforced character:
                        (!opts.firstMustMatch ||
                            charsToCheckCount !== whatToMatchVal.length)) {
                        charsMatchedTotal++;
                        // bail early if there's whitespace in front, imagine:
                        // abc important}
                        //   ^
                        //  start, match ["!important"], matchRightIncl()
                        //
                        // in case above, "c" consumed 1 patience, let's say 1 is left,
                        // we stumble upon "i" where "!" is missing. "c" is false start.
                        if (whitespaceInFrontOfFirstChar()) {
                            return false;
                        }
                        charsToCheckCount -= 2;
                        somethingFound = true;
                        break;
                    }
                    else if (nextCharInSource &&
                        nextCharToCompareAgainst &&
                        ((!opts.i && nextCharInSource === nextCharToCompareAgainst) ||
                            (opts.i &&
                                nextCharInSource.toLowerCase() ===
                                    nextCharToCompareAgainst.toLowerCase())) &&
                        // ensure we're not skipping the first enforced character:
                        (!opts.firstMustMatch ||
                            charsToCheckCount !== whatToMatchVal.length)) {
                        if (!charsMatchedTotal && !opts.hungry) {
                            return false;
                        }
                        charsToCheckCount -= 1;
                        somethingFound = true;
                        break;
                    }
                    else if (nextCharToCompareAgainst === undefined &&
                        patience >= 0 &&
                        somethingFound &&
                        (!opts.firstMustMatch || firstCharacterMatched) &&
                        (!opts.lastMustMatch || lastCharacterMatched)) {
                        // If "nextCharToCompareAgainst" is undefined, this
                        // means there are no more characters left to match,
                        // this is the last character to be matched.
                        // This means, if patience >= 0, this is it,
                        // the match is still positive.
                        return i;
                    }
                    // ███████████████████████████████████████
                }
                if (!somethingFound) {
                    // if the character was rogue, we mark it:
                    lastWasMismatched = i;
                    // patience--;
                    // console.log(
                    //   `350 ${`\u001b[${32}m${`SET`}\u001b[${39}m`} ${`\u001b[${33}m${`patience`}\u001b[${39}m`} = ${patience}`
                    // );
                }
            }
            else if (i === 0 &&
                charsToCheckCount === 1 &&
                !opts.lastMustMatch &&
                atLeastSomethingWasMatched) {
                return 0;
            }
            else {
                return false;
            }
        }
        // turn off "lastWasMismatched" if it's on and it hasn't been activated
        // on this current index:
        if (lastWasMismatched !== false && lastWasMismatched !== i) {
            lastWasMismatched = false;
        }
        // if all was matched, happy days
        if (charsToCheckCount < 1) {
            return i;
        }
        // iterate onto the next index, otherwise while would loop infinitely
        i = getNextIdx(i);
    }
    if (charsToCheckCount > 0) {
        if (special && whatToMatchValVal === "EOL") {
            return true;
        }
        if (opts &&
            opts.maxMismatches >= charsToCheckCount &&
            atLeastSomethingWasMatched) {
            return lastWasMismatched || 0;
        }
        return false;
    }
}
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
// Real deal
function main(mode, str, position, originalWhatToMatch, originalOpts) {
    // insurance
    if (isObj(originalOpts) &&
        Object.prototype.hasOwnProperty.call(originalOpts, "trimBeforeMatching") &&
        typeof originalOpts.trimBeforeMatching !== "boolean") {
        throw new Error(`string-match-left-right/${mode}(): [THROW_ID_09] opts.trimBeforeMatching should be boolean!${Array.isArray(originalOpts.trimBeforeMatching)
            ? ` Did you mean to use opts.trimCharsBeforeMatching?`
            : ""}`);
    }
    const opts = { ...defaults, ...originalOpts };
    if (typeof opts.trimCharsBeforeMatching === "string") {
        // arrayiffy if needed:
        opts.trimCharsBeforeMatching = arrayiffy(opts.trimCharsBeforeMatching);
    }
    // stringify all:
    opts.trimCharsBeforeMatching = opts.trimCharsBeforeMatching.map((el) => (isStr(el) ? el : String(el)));
    if (!isStr(str)) {
        return false;
    }
    if (!str.length) {
        return false;
    }
    if (!Number.isInteger(position) || position < 0) {
        throw new Error(`string-match-left-right/${mode}(): [THROW_ID_03] the second argument should be a natural number. Currently it's of a type: ${typeof position}, equal to:\n${JSON.stringify(position, null, 4)}`);
    }
    let whatToMatch;
    let special;
    if (isStr(originalWhatToMatch)) {
        whatToMatch = [originalWhatToMatch];
    }
    else if (Array.isArray(originalWhatToMatch)) {
        whatToMatch = originalWhatToMatch;
    }
    else if (!originalWhatToMatch) {
        whatToMatch = originalWhatToMatch;
    }
    else if (typeof originalWhatToMatch === "function") {
        whatToMatch = [];
        whatToMatch.push(originalWhatToMatch);
    }
    else {
        throw new Error(`string-match-left-right/${mode}(): [THROW_ID_05] the third argument, whatToMatch, is neither string nor array of strings! It's ${typeof originalWhatToMatch}, equal to:\n${JSON.stringify(originalWhatToMatch, null, 4)}`);
    }
    if (originalOpts && !isObj(originalOpts)) {
        throw new Error(`string-match-left-right/${mode}(): [THROW_ID_06] the fourth argument, options object, should be a plain object. Currently it's of a type "${typeof originalOpts}", and equal to:\n${JSON.stringify(originalOpts, null, 4)}`);
    }
    let culpritsIndex = 0;
    let culpritsVal = "";
    if (opts &&
        opts.trimCharsBeforeMatching &&
        opts.trimCharsBeforeMatching.some((el, i) => {
            if (el.length > 1) {
                culpritsIndex = i;
                culpritsVal = el;
                return true;
            }
            return false;
        })) {
        throw new Error(`string-match-left-right/${mode}(): [THROW_ID_07] the fourth argument, options object contains trimCharsBeforeMatching. It was meant to list the single characters but one of the entries at index ${culpritsIndex} is longer than 1 character, ${culpritsVal.length} (equals to ${culpritsVal}). Please split it into separate characters and put into array as separate elements.`);
    }
    // action
    // CASE 1. If it's driven by callback-only, the 3rd input argument, what to look
    // for - is falsey - empty string within array (or not), OR given null
    if (!whatToMatch ||
        !Array.isArray(whatToMatch) || // 0
        (Array.isArray(whatToMatch) && !whatToMatch.length) || // []
        (Array.isArray(whatToMatch) &&
            whatToMatch.length === 1 &&
            isStr(whatToMatch[0]) &&
            !whatToMatch[0].trim()) // [""]
    ) {
        if (typeof opts.cb === "function") {
            let firstCharOutsideIndex;
            // matchLeft() or matchRightIncl() methods start at index "position"
            let startingPosition = position;
            if (mode === "matchLeftIncl" || mode === "matchRight") {
                startingPosition += 1;
            }
            if (mode[5] === "L") {
                for (let y = startingPosition; y--;) {
                    // assemble the value of the current character
                    const currentChar = str[y];
                    // do the actual evaluation, is the current character non-whitespace/non-skiped
                    if ((!opts.trimBeforeMatching ||
                        (opts.trimBeforeMatching &&
                            currentChar !== undefined &&
                            currentChar.trim())) &&
                        (!opts.trimCharsBeforeMatching ||
                            !opts.trimCharsBeforeMatching.length ||
                            (currentChar !== undefined &&
                                !opts.trimCharsBeforeMatching.includes(currentChar)))) {
                        firstCharOutsideIndex = y;
                        break;
                    }
                }
            }
            else if (mode.startsWith("matchRight")) {
                for (let y = startingPosition; y < str.length; y++) {
                    // assemble the value of the current character
                    const currentChar = str[y];
                    // do the actual evaluation, is the current character non-whitespace/non-skiped
                    if ((!opts.trimBeforeMatching ||
                        (opts.trimBeforeMatching && currentChar.trim())) &&
                        (!opts.trimCharsBeforeMatching ||
                            !opts.trimCharsBeforeMatching.length ||
                            !opts.trimCharsBeforeMatching.includes(currentChar))) {
                        firstCharOutsideIndex = y;
                        break;
                    }
                }
            }
            if (firstCharOutsideIndex === undefined) {
                return false;
            }
            const wholeCharacterOutside = str[firstCharOutsideIndex];
            const indexOfTheCharacterAfter = firstCharOutsideIndex + 1;
            let theRemainderOfTheString = "";
            if (indexOfTheCharacterAfter && indexOfTheCharacterAfter > 0) {
                theRemainderOfTheString = str.slice(0, indexOfTheCharacterAfter);
            }
            if (mode[5] === "L") {
                return opts.cb(wholeCharacterOutside, theRemainderOfTheString, firstCharOutsideIndex);
            }
            // ELSE matchRight & matchRightIncl
            if (firstCharOutsideIndex && firstCharOutsideIndex > 0) {
                theRemainderOfTheString = str.slice(firstCharOutsideIndex);
            }
            return opts.cb(wholeCharacterOutside, theRemainderOfTheString, firstCharOutsideIndex);
        }
        let extraNote = "";
        if (!originalOpts) {
            extraNote =
                " More so, the whole options object, the fourth input argument, is missing!";
        }
        throw new Error(`string-match-left-right/${mode}(): [THROW_ID_08] the third argument, "whatToMatch", was given as an empty string. This means, you intend to match purely by a callback. The callback was not set though, the opts key "cb" is not set!${extraNote}`);
    }
    // Case 2. Normal operation where callback may or may not be present, but it is
    // only accompanying the matching of what was given in 3rd input argument.
    // Then if 3rd arg's contents were matched, callback is checked and its Boolean
    // result is merged using logical "AND" - meaning both have to be true to yield
    // final result "true".
    for (let i = 0, len = whatToMatch.length; i < len; i++) {
        special = typeof whatToMatch[i] === "function";
        // since input can be function, we need to grab the value explicitly:
        const whatToMatchVal = whatToMatch[i];
        let fullCharacterInFront;
        let indexOfTheCharacterInFront;
        let restOfStringInFront = "";
        let startingPosition = position;
        if (mode === "matchRight") {
            startingPosition += 1;
        }
        else if (mode === "matchLeft") {
            startingPosition -= 1;
        }
        const found = march(str, startingPosition, whatToMatchVal, opts, special, (i2) => (mode[5] === "L" ? i2 - 1 : i2 + 1));
        // if march() returned positive result and it was "special" case,
        // Bob's your uncle, here's the result:
        if (found &&
            special &&
            typeof whatToMatchVal === "function" &&
            whatToMatchVal() === "EOL") {
            return whatToMatchVal() &&
                (opts.cb
                    ? opts.cb(fullCharacterInFront, restOfStringInFront, indexOfTheCharacterInFront)
                    : true)
                ? whatToMatchVal()
                : false;
        }
        // now, the "found" is the index of the first character of what was found.
        // we need to calculate the character to the left/right of it:
        if (Number.isInteger(found)) {
            indexOfTheCharacterInFront = mode.startsWith("matchLeft")
                ? found - 1
                : found + 1;
            //
            if (mode[5] === "L") {
                restOfStringInFront = str.slice(0, found);
            }
            else {
                restOfStringInFront = str.slice(indexOfTheCharacterInFront);
            }
        }
        if (indexOfTheCharacterInFront < 0) {
            indexOfTheCharacterInFront = undefined;
        }
        if (str[indexOfTheCharacterInFront]) {
            fullCharacterInFront = str[indexOfTheCharacterInFront];
        }
        if (Number.isInteger(found) &&
            (opts.cb
                ? opts.cb(fullCharacterInFront, restOfStringInFront, indexOfTheCharacterInFront)
                : true)) {
            return whatToMatchVal;
        }
    }
    return false;
}
// External API functions
function matchLeftIncl(str, position, whatToMatch, opts) {
    return main("matchLeftIncl", str, position, whatToMatch, opts);
}
function matchLeft(str, position, whatToMatch, opts) {
    return main("matchLeft", str, position, whatToMatch, opts);
}
function matchRightIncl(str, position, whatToMatch, opts) {
    return main("matchRightIncl", str, position, whatToMatch, opts);
}
function matchRight(str, position, whatToMatch, opts) {
    return main("matchRight", str, position, whatToMatch, opts);
}

exports.matchLeft = matchLeft;
exports.matchLeftIncl = matchLeftIncl;
exports.matchRight = matchRight;
exports.matchRightIncl = matchRightIncl;

Object.defineProperty(exports, '__esModule', { value: true });

})));
